import os
import sys
import wave
import subprocess
import math
import numpy as np
from PIL import Image, ImageDraw, ImageFont
from pathlib import Path
from tqdm import tqdm

# =================================================================
# CONFIGURATION
# =================================================================
class Config:
    ASSET_DIR = './assets_ready'
    OUTPUT_FILE = 'demoreel.mp4'
    AUDIO_FILE = 'music.wav'
    
    # Render Settings
    FPS = 30
    res_x, res_y = 1920, 1080
    WINDOW_SIZE = (res_x, res_y)
    
    # Physics Tuning (Matched to V6 Commander)
    GRAVITY = 0.8
    FRICTION = 0.90
    SMOOTHING = 0.15
    
    VOLUME_SCALE_STRENGTH = 0.3
    VOLUME_JUMP_STRENGTH = -25
    BASS_THUMP_STRENGTH = 15
    TREBLE_JITTER_STRENGTH = 20
    ROTATION_STRENGTH = 15

    # Visuals
    FALLBACK_COLOR = (20, 20, 20) 
    TEXT_COLOR = (255, 255, 255)

# =================================================================
# UTILS
# =================================================================
def get_background():
    """Smartly finds the background image."""
    options = ['background.png', 'background.jpg', 'background.jpeg']
    for opt in options:
        if os.path.exists(opt):
            print(f" [System] Found background: {opt}")
            try:
                img = Image.open(opt).convert("RGBA")
                return img.resize(Config.WINDOW_SIZE, Image.Resampling.LANCZOS)
            except:
                pass
    print(" [System] No background found. Using solid color.")
    return Image.new("RGBA", Config.WINDOW_SIZE, Config.FALLBACK_COLOR)

def load_audio_data(filename):
    if not os.path.exists(filename):
        print(f"ERROR: {filename} not found! Please place a .wav file in this folder.")
        sys.exit(1)
    try:
        wf = wave.open(filename, 'rb')
        params = wf.getparams()
        n_channels, sampwidth, framerate, n_frames = params[:4]
        str_data = wf.readframes(n_frames)
        wf.close()
        
        # Convert to normalized float (-1.0 to 1.0)
        dtype = np.int16 if sampwidth == 2 else np.uint8
        audio = np.frombuffer(str_data, dtype=dtype)
        if n_channels == 2: audio = audio[::2]
        
        norm_factor = float(2**(8*sampwidth - 1))
        return audio / norm_factor, framerate, n_frames / framerate
    except Exception as e:
        print(f"Error reading audio: {e}")
        sys.exit(1)

def get_audio_features(audio_data, frame_idx, fps, sample_rate):
    center = int((frame_idx / fps) * sample_rate)
    win = 2048
    start = max(0, center - win // 2)
    end = min(len(audio_data), center + win // 2)
    chunk = audio_data[start:end]
    
    if len(chunk) < win: return {'vol':0, 'bass':0, 'treble':0}
    
    # Analyze Float Audio
    vol = np.sqrt(np.mean(chunk**2))
    
    fft_mags = np.abs(np.fft.rfft(chunk))
    freqs = np.fft.rfftfreq(len(chunk), 1.0/sample_rate)
    
    bass = np.sum(fft_mags[np.where((freqs >= 50) & (freqs <= 250))])
    treble = np.sum(fft_mags[np.where((freqs >= 2000) & (freqs <= 5000))])
    
    # Multipliers tuned for Normalized Float Audio
    return {
        'vol': np.clip(vol * 5, 0, 1),
        'bass': np.clip(bass / 100, 0, 1),
        'treble': np.clip(treble / 100, 0, 1)
    }

# =================================================================
# MAIN RENDERER
# =================================================================
def main():
    print(f"--- GENERATING DEMO REEL ({Config.OUTPUT_FILE}) ---")
    
    # 1. Load Resources
    audio_data, sample_rate, duration = load_audio_data(Config.AUDIO_FILE)
    bg_master = get_background()
    
    if not os.path.exists(Config.ASSET_DIR): return
    avatars = sorted([d for d in os.listdir(Config.ASSET_DIR) if os.path.isdir(os.path.join(Config.ASSET_DIR, d))])
    if not avatars: print("No avatars found!"); return
    print(f"Found {len(avatars)} avatars.")

    # 2. Setup FFMPEG
    cmd = [
        'ffmpeg', '-y', '-f', 'rawvideo', '-vcodec', 'rawvideo',
        '-s', f'{Config.res_x}x{Config.res_y}', '-pix_fmt', 'rgba',
        '-r', str(Config.FPS), '-i', '-', '-i', Config.AUDIO_FILE,
        '-map', '0:v', '-map', '1:a', '-c:v', 'libx264',
        '-pix_fmt', 'yuv420p', '-preset', 'fast', '-shortest',
        Config.OUTPUT_FILE
    ]
    proc = subprocess.Popen(cmd, stdin=subprocess.PIPE)
    
    # 3. Physics Init
    total_frames = int(duration * Config.FPS)
    frames_per_av = total_frames // len(avatars)
    
    pos_y = Config.WINDOW_SIZE[1] * 0.9
    vel_y, vel_x = 0, 0
    scale, angle = 1.0, 0.0
    
    curr_idx = -1
    curr_img = None
    
    print(f"Rendering {total_frames} frames...")
    
    try:
        for f in tqdm(range(total_frames), unit="fr"):
            
            # Switch Avatar logic
            idx = min(f // frames_per_av, len(avatars) - 1)
            if idx != curr_idx:
                curr_idx = idx
                path = os.path.join(Config.ASSET_DIR, avatars[idx], "avatar.png")
                raw = Image.open(path).convert("RGBA")
                # Resize avatar to 80% screen height
                tgt_h = int(Config.WINDOW_SIZE[1] * 0.8)
                raw = raw.resize((int(raw.width * (tgt_h/raw.height)), tgt_h), Image.Resampling.LANCZOS)
                curr_img = raw
                # Transition bounce
                vel_y = -30
                pos_y = Config.WINDOW_SIZE[1] * 0.9
                scale = 0.5 # Shrink for pop-in

            # Physics Update
            aud = get_audio_features(audio_data, f, Config.FPS, sample_rate)
            
            jit_dir = 1 if f % 4 < 2 else -1
            
            vel_y += (aud['vol'] * Config.VOLUME_JUMP_STRENGTH) + (aud['bass'] * Config.BASS_THUMP_STRENGTH) + Config.GRAVITY
            vel_x += (aud['treble'] * Config.TREBLE_JITTER_STRENGTH * jit_dir)
            vel_y *= Config.FRICTION
            vel_x *= Config.FRICTION
            
            pos_y += vel_y
            floor = Config.WINDOW_SIZE[1] * 0.95
            if pos_y > floor: pos_y = floor; vel_y = 0
            
            tgt_scale = 1.0 + (aud['vol'] * Config.VOLUME_SCALE_STRENGTH)
            tgt_angle = (aud['bass'] - 0.5) * Config.ROTATION_STRENGTH
            
            scale += (tgt_scale - scale) * Config.SMOOTHING
            angle += (tgt_angle - angle) * Config.SMOOTHING
            
            # Composite Frame
            frame = bg_master.copy()
            
            # Transform
            av = curr_img.rotate(angle, expand=True, resample=Image.BICUBIC)
            av = av.resize((int(av.width*scale), int(av.height*scale)), Image.BICUBIC)
            
            # Paste
            draw_x = int((Config.WINDOW_SIZE[0] - av.width)/2 + vel_x)
            draw_y = int(pos_y - av.height)
            frame.paste(av, (draw_x, draw_y), av)
            
            # Text
            draw = ImageDraw.Draw(frame)
            try: font = ImageFont.truetype("DejaVuSans-Bold.ttf", 60)
            except: font = ImageFont.load_default()
            
            txt = avatars[curr_idx].replace('_', ' ')
            tx, ty = 50, Config.WINDOW_SIZE[1] - 100
            
            # Shadow
            draw.text((tx+4, ty+4), txt, fill=(0,0,0), font=font) 
            # Main Text
            draw.text((tx, ty), txt, fill=Config.TEXT_COLOR, font=font)
            
            proc.stdin.write(frame.tobytes())
            
    except KeyboardInterrupt:
        print("Aborted.")
    
    proc.stdin.close()
    proc.wait()
    print("Done.")

if __name__ == "__main__":
    main()
